/**
 * AmbientTalk/2 Project
 * SerializationTest.java created on 28-dec-2006 at 19:48:13
 * (c) Programming Technology Lab, 2006 - 2007
 * Authors: Tom Van Cutsem & Stijn Mostinckx
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */
package edu.vub.at.actors.natives;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Proxy;
import java.util.Vector;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.minlog.Log;
import com.sun.org.apache.xpath.internal.operations.Variable;
import com.sun.xml.internal.messaging.saaj.util.ByteInputStream;

import edu.vub.at.AmbientTalkTest;
import edu.vub.at.actors.id.ATObjectID;
import edu.vub.at.actors.id.ActorID;
import edu.vub.at.actors.id.VirtualMachineID;
import edu.vub.at.actors.net.cmd.CMDHandshake;
import edu.vub.at.actors.net.cmd.CMDJoinServices;
import edu.vub.at.actors.net.cmd.CMDTransmitATMessage;
import edu.vub.at.eval.Evaluator;
import edu.vub.at.exceptions.InterpreterException;
import edu.vub.at.exceptions.XIOProblem;
import edu.vub.at.kryo.ATKryo;
import edu.vub.at.kryo.ThreadATKryoContext;
import edu.vub.at.objects.ATAbstractGrammar;
import edu.vub.at.objects.ATBoolean;
import edu.vub.at.objects.ATObject;
import edu.vub.at.objects.ATTable;
import edu.vub.at.objects.ATTypeTag;
import edu.vub.at.objects.coercion.Coercer;
import edu.vub.at.objects.coercion.NativeTypeTags;
import edu.vub.at.objects.natives.FieldMap;
import edu.vub.at.objects.natives.MethodDictionary;
import edu.vub.at.objects.natives.NATClosure;
import edu.vub.at.objects.natives.NATContext;
import edu.vub.at.objects.natives.NATMethod;
import edu.vub.at.objects.natives.NATObject;
import edu.vub.at.objects.natives.NATTable;
import edu.vub.at.objects.natives.NATText;
import edu.vub.at.objects.natives.NATTypeTag;
import edu.vub.at.objects.natives.NativeATObject;
import edu.vub.at.objects.natives.grammar.AGAssignField;
import edu.vub.at.objects.natives.grammar.AGSymbol;
import edu.vub.at.parser.NATParser;
import edu.vub.util.MultiMap;

/**
 * A test case for object serialization.
 *
 * @author tvcutsem
 */
public class SerializationTest extends AmbientTalkTest {
	
	private static byte[] serialize(Object o) throws IOException {
		ATKryo kryo = ThreadATKryoContext.get();
		ByteArrayOutputStream buffer = new ByteArrayOutputStream();
		Output out = new Output(buffer);
		// Do the initial write replace here:
		
		o = kryo.writeReplace(o);
		kryo.getKryo().writeClassAndObject(out, o);
		out.flush();
		out.close();
		return buffer.toByteArray();
	}
	
	private static Object deserialize(byte[] b) throws IOException, ClassNotFoundException {
		ATKryo kryo = ThreadATKryoContext.get();
		ByteArrayInputStream in = new ByteArrayInputStream(b);
		Input input = new Input(in);
		
		Object readObject = kryo.getKryo().readClassAndObject(input);
		readObject = kryo.readResolve(readObject);
		
		return readObject;
	}
	
	private static Object copy(Object o) throws IOException, ClassNotFoundException {
		return deserialize(serialize(o));
	}
	
	public void testTXTSerialization() throws IOException, ClassNotFoundException {
		NATText boeTXT = NATText.atValue("boe");
		Object cpy = copy(boeTXT);
		NATText boeTXT2 = (NATText) cpy; 
		assertEquals(boeTXT.toString(), boeTXT2.toString());
	}
	
	public void testPacketTXTSerialization() throws IOException, ClassNotFoundException, InterpreterException {
		NATText boeTXT = NATText.atValue("boe");
		Packet p = new Packet("test", boeTXT);
		NATText obj = (NATText) p.unpack();
		assertEquals(boeTXT.toString(), obj.toString());
	}
	
	public void testAGSerialization() throws IOException, ClassNotFoundException {
		NATText boeTXT = NATText.atValue("boe");
		AGSymbol fooSYM = AGSymbol.jAlloc("foo");
		AGAssignField ass = new AGAssignField(boeTXT, fooSYM, boeTXT);
		AGAssignField ass2 = (AGAssignField) copy(ass);
		assertEquals(ass.toString(), ass2.toString());
		
		Log.warn(ass.base_fieldName().toString()+ "  " + ass2.base_fieldName().toString());
		assertTrue(ass.base_fieldName() == ass2.base_fieldName()); //TODO 
	}
	
	public void testParseTreeSerialization() throws InterpreterException {
		ATAbstractGrammar ag = NATParser.parse("test", "{ |x,y| x + y }");
		Packet p = new Packet("test", ag);
		ATObject o = p.unpack();
		assertEquals(ag.toString(), o.toString());
		assertFalse(ag == o);
	}
	
	/**
	 * Tests whether a coercer is correctly serialized such that upon
	 * deserialization it still holds that the deserialized value is an instance
	 * of the given Java type.
	 */
	public void testCoercerSerialization() throws InterpreterException { //Needs the resolveProxyClass
		NATObject isolate = new NATObject(new ATTypeTag[] { NativeTypeTags._ISOLATE_, NativeTypeTags._TABLE_ });
		ATTable coercer = isolate.asTable();
		assertTrue(coercer instanceof ATTable);
		Packet p = new Packet("test", coercer);
		ATObject obj = p.unpack();
		assertTrue(obj instanceof ATTable);
		assertFalse(obj == isolate);
	}
	
	public void testMinimalCoercerObjectSerialisation() throws InterpreterException {
		NATObject bool = new NATObject(new ATTypeTag[] {NativeTypeTags._BOOLEAN_});
		ATBoolean coercer = bool.asBoolean();
		Packet p = new Packet("test", coercer);
		ATObject obj = p.unpack();
	}
	
	public void testMinimalCoercerObject2Serialisation() throws InterpreterException {
		NATText boeTXT = NATText.atValue("boe");
		Object o = Coercer.coerce(boeTXT, NATText.class);
		Packet p = new Packet("test", (ATObject) o);
		ATObject obj = p.unpack();
		Log.warn(((NATText) obj).javaValue);
	}
	
	public void testIsolateSerialisation() throws InterpreterException, ClassNotFoundException, IOException {
		NATObject isolate = new NATObject(new ATTypeTag[] { NativeTypeTags._ISOLATE_, NativeTypeTags._TABLE_ });
		Packet p = new Packet("test", isolate);
		assertEquals(isolate.toString(), p.unpack().toString());
	}
	
	public void testGlobalLexicalScopeForIsolate() throws InterpreterException {
		NATObject lexicalParent_ = Evaluator.getGlobalLexicalScope();
		Packet p = new Packet("test", lexicalParent_);
		ATObject obj = p.unpack();
	}
	
	public void testNATTableEMPTY() throws InterpreterException{
		NATTable natEmpty = NATTable.EMPTY;
		Packet p = new Packet("test", natEmpty);
		NATTable obj = (NATTable) p.unpack();
	}
	
	
	public void testTypeTags() throws InterpreterException, ClassNotFoundException, IOException{
		ATTypeTag[] payload = new ATTypeTag[] { NativeTypeTags._ISOLATE_, NativeTypeTags._TABLE_, NativeTypeTags._FARREF_};
		copy(payload);
	}
	
	public void testIsolateAsTableSerialisation() throws InterpreterException, NoSuchFieldException, SecurityException, IllegalArgumentException, IllegalAccessException {
		NATObject isolate = new NATObject(new ATTypeTag[] { NativeTypeTags._ISOLATE_, NativeTypeTags._TABLE_ });
		ATTable coercer = isolate.asTable();
		Coercer c = (Coercer) ((Proxy) coercer).getInvocationHandler(coercer);
		Field f = c.getClass().getDeclaredField("principal_");
		f.setAccessible(true);
		ATObject principal_ = (ATObject) f.get(c);
		Packet p = new Packet("test", principal_);
		ATObject obj = p.unpack();
		assertEquals(obj.toString(), coercer.toString());
	}
	
	private ELActor setUpActor(ELVirtualMachine host) throws InterpreterException {
		return host.createEmptyActor().getFarHost();
	}
	
	public void testParsedCode1() throws InterpreterException{

	    String _TEST_GROUP_NAME_ = "AmbientTalkTest";
		
		ELVirtualMachine virtual1_ = new ELVirtualMachine(Evaluator.getNil(), new SharedActorField[] { }, _TEST_GROUP_NAME_, ELVirtualMachine._DEFAULT_IP_ADDRESS_, System.out);
		ELActor subscriber = setUpActor(virtual1_);
		
	ATAbstractGrammar grm = NATParser.parse("Point Example",

				"def Point := object: { \n" +
				"	def x := 0; \n" +
				"	def y := 0; \n" +
				"   def init(aX, aY) { \n" +
				"		x := ax; \n" +
				"		x := ay; \n" +
				"	}; \n" +
				" 	def sumOfSquares() {x*x + y*y}; \n" +
				"}");					
		
		ATObject payload = subscriber.sync_event_eval(grm);
		
		Log.warn(payload.toString());
		Log.warn(payload.getClass().getName());
		
		Packet p = new Packet(payload);
		assertEquals(p.unpack(), payload);
		
	}
	
	public static class TableHolder {
		final ATTable table_;
		public Object o;
		public TableHolder(ATTable t) { table_ = t; }
		public TableHolder(){
			table_ = null;
		}
	}
	
	public void testNattableEmpty() throws InterpreterException, ClassNotFoundException, IOException {
		TableHolder th = new  TableHolder(NATTable.EMPTY);
		th.o = new Vector<Integer>();
		copy(th);
	}
	
	public void testParsedCode2() throws InterpreterException{

	    String _TEST_GROUP_NAME_ = "AmbientTalkTest";
		
		ELVirtualMachine virtual1_ = new ELVirtualMachine(Evaluator.getNil(), new SharedActorField[] { }, _TEST_GROUP_NAME_, ELVirtualMachine._DEFAULT_IP_ADDRESS_, System.out);
		ELActor subscriber = setUpActor(virtual1_);
		
		ATAbstractGrammar grm = NATParser.parse("Lambda Example",
				"{|a, b| a+ b}(3,2)");
				
		
		ATObject payload = subscriber.sync_event_eval(grm);
		
		Packet p = new Packet(payload);
		Log.warn(payload.toString());
		Log.warn(p.unpack().toString());
		assertEquals(payload, p.unpack());
		
	}
	
	public void testParsedCode3() throws InterpreterException{

	    String _TEST_GROUP_NAME_ = "AmbientTalkTest";
		
		ELVirtualMachine virtual1_ = new ELVirtualMachine(Evaluator.getNil(), new SharedActorField[] { }, _TEST_GROUP_NAME_, ELVirtualMachine._DEFAULT_IP_ADDRESS_, System.out);
		ELActor subscriber = setUpActor(virtual1_);
		
		ATAbstractGrammar grm = NATParser.parse("Array Example",
			"def Enumerable := object: { \n" +
			"  def collect: closure { \n" +
			"    def c := self.new([]); \n" +
			"    self.each: { |v| \n" +
			"      c.add(closure(v)); \n" +
			"    }; \n" +
			"    c; \n" +
			"  }; \n" +
		"	}; \n" +
		"	def Array := object: { \n" +
		"	  def elements := []; \n" +
		"	  def init(a) { elements := a; }; \n" +
		"	  def add(v) { elements := elements + [v]; self }; \n" +
		"	  def collect: closure { \n" +
		"	    Enumerable^collect: closure; \n" +
		"	  }; \n" +
		"	  def each: clo { \n" +
		"	    1.to: elements.length + 1 do: { |i| \n" +
		"	      clo(elements[i]); \n" +
		"	    }; \n" +
		"	  }; \n" +
		"	}; \n" +
		
		" Array.add(1).add(2).add(3); \n" +
		" def c := Array.collect: { |v| v+1 }; \n" +
		" c.each: { |v| v}; \n" +
		" \"ok\" \n"
				);
				
		
		ATObject payload = subscriber.sync_event_eval(grm);
		
		Packet p = new Packet(payload);

		assertEquals(payload, p.unpack());
		
	}
	
	public void testParsedCode4() throws InterpreterException{

	    String _TEST_GROUP_NAME_ = "AmbientTalkTest";
		
		ELVirtualMachine virtual1_ = new ELVirtualMachine(Evaluator.getNil(), new SharedActorField[] { }, _TEST_GROUP_NAME_, ELVirtualMachine._DEFAULT_IP_ADDRESS_, System.out);
		ELActor subscriber = setUpActor(virtual1_);
		
		ATAbstractGrammar grm = NATParser.parse("Bank Example",
			"def makeBankAccount(balance) { \n" +
			"  object: { \n" +
			"    def deposit(amnt) { \n" +
			"      balance := balance + amnt; \n" +
			"      \"ok\" \n" +
			"    }; \n" +
			"  } \n" +
			"}; \n" +
			
			"def b := makeBankAccount(100); \n" + 
			"b.deposit(10);");
							
		
		ATObject payload = subscriber.sync_event_eval(grm);
		
		Packet p = new Packet(payload);

		assertEquals(payload, p.unpack());
		
	}
	
	public void testParsedCode5() throws InterpreterException, ClassNotFoundException, IOException{

	    String _TEST_GROUP_NAME_ = "AmbientTalkTest";
		
		ELVirtualMachine virtual1_ = new ELVirtualMachine(Evaluator.getNil(), new SharedActorField[] { }, _TEST_GROUP_NAME_, ELVirtualMachine._DEFAULT_IP_ADDRESS_, System.out);
		ELActor subscriber = setUpActor(virtual1_);
		
		ATAbstractGrammar grm = 		
						NATParser.parse("DistributionTest#setUpConnectionObservers()",
								"deftype Service; \n" +
								"export: (object: { nil }) as: Service");
							
		
		ATObject payload = subscriber.sync_event_eval(grm);
		
		Packet p = new Packet(payload);

		assertEquals(payload, p.unpack());
		
	}
	

	public void testCMDHandshake() throws ClassNotFoundException, IOException{
		CMDHandshake cmd = new CMDHandshake(new VirtualMachineID());
		copy(cmd);
	}
	
	public void testCMDTransmitATMessage() throws ClassNotFoundException, IOException, InterpreterException{
		NATObject isolate = new NATObject(new ATTypeTag[] { NativeTypeTags._ISOLATE_, NativeTypeTags._TABLE_ });
		ATTable coercer = isolate.asTable();
		assertTrue(coercer instanceof ATTable);
		Packet p = new Packet("test", coercer);
		
		CMDTransmitATMessage cmd = new CMDTransmitATMessage(new ActorID(), p);
		copy(cmd);
	}
	
	public void testIDSerialization() throws ClassNotFoundException, IOException{
		ATObjectID id = new ATObjectID(new VirtualMachineID(), new ActorID(), "test");
		assertTrue(id.equals(copy(id)));
	}
	
	
	public static abstract class TEST implements Serializable {
		public abstract Object writeReplace();
		public abstract Object readResolve();
	}
	
	
	public static class st implements Runnable {
		public void run() {
			try {
				serialize(new A());
				serialize(new A2());
				new Packet("test", NATTable.EMPTY);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (XIOProblem e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}
	
	public static class A extends TEST {
		Integer a = 2;
		
		public A(){
			a = null;
		}
		
		public A(Integer a){
			this.a = a;
		}
		
		public Object writeReplace(){
			//new Thread(new st()).start();
			System.out.println("k wr");
			return new A2(3, 2.5, "bla");
		}

		public Object readResolve() {
			System.out.println("mag blijkbaar wel");
			return this;
		}
		
	}
	
	public static class A2 extends TEST{
		Integer a = 3;
		Double b = 2.5;
		String c = "bla";
		
		public A2(){
			a = null;
			b = null;
			c = "";
		}
		
		public A2(Integer a, Double b, String c){
			this.a = a;
			this.b = b;
			this.c = c;
		}
		
		public Object writeReplace(){
			System.out.println("mag blijkbaar wel");
			return new A3(3);
		}

		public Object readResolve() {
			System.out.println("k rr");
			return new A(2);
		}
	}
	
	public static class A3 extends TEST{
		Integer a = 3;

		
		public A3(){
			a = null;
		}
		
		public A3(Integer a){
			this.a = a;
		}
		
		public Object writeReplace(){
			System.out.println("mag nt");
			return this;
		}

		public Object readResolve() {
			System.out.println("k rr");
			return new A(2);
		}
	}
	
	public static class Payload implements Serializable{
		public Object a;
		public Object b;
		public Object c;
		
		public Payload() {
			a = new A();
			b = a;
			c = new A();
		}
	}
	
	public void testWriteReplace() throws ClassNotFoundException, IOException{
		Payload payload = new Payload();
		byte[] b = serialize(payload);
		Object copy = deserialize(b);
		
		System.out.println(((A)((Payload)copy).a).a.toString());
	}
	
	
	public byte[] serialize2(Object o) throws IOException{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ObjectOutputStream oout = new ObjectOutputStream(out);
		oout.writeObject(o);
		return out.toByteArray();
	}
	
	public Object deserialize2(byte[] b) throws ClassNotFoundException, IOException{
		ByteArrayInputStream in = new ByteArrayInputStream(b);
		ObjectInputStream oin = new ObjectInputStream(in);
		return 	oin.readObject();
	}
	
	public void testWriteReplaceJava() throws ClassNotFoundException, IOException{
		Payload payload = new Payload();
		byte[] b = serialize2(payload);
		Object copy = deserialize2(b);
		
		System.out.println(((A)((Payload)copy).a).a.toString());

	}
	
	public void testMultiMapSerialization() throws InterpreterException, ClassNotFoundException, IOException {
		MultiMap mp = new MultiMap();
		mp.put(new Integer(1), "aa");
		mp.put(new Integer(1), "ab");
		mp.put(new Integer(1), new Double(1.1));
		mp.put(new Integer(2), "dd");
		mp.put(new Integer(3), "bla bla");
		mp.put(new Integer(3), "bla");
		MultiMap copy = (MultiMap) copy(mp);
		assertTrue(mp.equals(copy));
	}
	
	public void testMethodDictionarySerialization() throws ClassNotFoundException, IOException{
		MethodDictionary md = new MethodDictionary();
		md.put(new Integer(1), "aa");
		md.put(new Integer(2), "dd");
		md.put(new Integer(3), "bla bla");
		MethodDictionary copy = (MethodDictionary) copy(md);
		assertTrue(md.get(new Integer(1)).equals(copy.get(new Integer(1))));
		assertTrue(md.get(new Integer(2)).equals(copy.get(new Integer(2))));
		assertTrue(md.get(new Integer(3)).equals(copy.get(new Integer(3))));
	}
	
	public void testFieldMapSerialization() throws ClassNotFoundException, IOException{
		FieldMap fm = new FieldMap();
		AGSymbol a = AGSymbol.jAlloc("haha");
		AGSymbol b = AGSymbol.jAlloc("hehe");
		AGSymbol c = AGSymbol.jAlloc("selector");
		fm.put(a);
		fm.put(b);
		fm.put(c);

		FieldMap copy = (FieldMap) copy(fm);
		
		assertEquals(fm.get(a), copy.get(a));
		assertEquals(fm.get(b), copy.get(b));
		assertEquals(fm.get(c), copy.get(c));
		
	}
	
	static class H {
		public transient Integer a;
		public Integer b = new Integer(2);
		public H(){
		}
	}
	
	public void testTransientSerialization() throws ClassNotFoundException, IOException{
		H h = new H();
		h.a = new Integer(1);
		assertNull(((H)copy(new H())).a);
	}

	/** Apparently, the payload generated by different packets is different even for the same object */
	public void testEquality() throws InterpreterException {
		ATObject obj = new NATObject();
		Packet p1 = new Packet(obj);
		Packet p2 = new Packet(obj);
		assertFalse(p1.equals(p2));
	}
	
	
}
